#include <stdio.h>
#include <conio.h>
#include <windows.h>
#include "cifxuser.h"
#include "cifxErrors.h"

#define PACKET_WAIT_TIMEOUT 20
#define IO_WAIT_TIMEOUT     10
#define HOSTSTATE_TIMEOUT   5000

/*****************************************************************************/
/*! Displays a hex dump on the debug console (16 bytes per line)
*   \param pbData     Pointer to dump data
*   \param ulDataLen  Length of data dump                                    */
/*****************************************************************************/
void DumpData(unsigned char* pbData, unsigned long ulDataLen)
{
  unsigned long ulIdx = 0;
  
  for(ulIdx = 0; ulIdx < ulDataLen; ++ulIdx)
  {
    if(0 == (ulIdx % 16))
      printf("\r\n");

    printf("%02X ", pbData[ulIdx]);
  }
  printf("\r\n");
}

/*****************************************************************************/
/*! Dumps a rcX packet to debug console
*   \param ptPacket Pointer to packed being dumped                           */
/*****************************************************************************/
void DumpPacket(CIFX_PACKET* ptPacket)
{
  printf("Dest   : 0x%08X      ID   : 0x%08X\r\n", ptPacket->tHeader.ulDest,   ptPacket->tHeader.ulId);
  printf("Src    : 0x%08X      Sta  : 0x%08X\r\n", ptPacket->tHeader.ulSrc,    ptPacket->tHeader.ulState);
  printf("DestID : 0x%08X      Cmd  : 0x%08X\r\n", ptPacket->tHeader.ulDestId, ptPacket->tHeader.ulCmd);
  printf("SrcID  : 0x%08X      Ext  : 0x%08X\r\n", ptPacket->tHeader.ulSrcId,  ptPacket->tHeader.ulExt);
  printf("Len    : 0x%08X      Rout : 0x%08X\r\n", ptPacket->tHeader.ulLen,    ptPacket->tHeader.ulRout);

  printf("Data:");
  DumpData(ptPacket->abData, ptPacket->tHeader.ulLen);
}

/*****************************************************************************/
/*! Function to demonstrate the board/channel enumeration
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
void EnumBoardDemo(HANDLE hDriver)
{
  uint32_t          ulBoard    = 0;
  BOARD_INFORMATION tBoardInfo = {0};

  printf("---------- Board/Channel enumeration demo ----------\r\n");


  /* Iterate over all boards */
  while(CIFX_NO_ERROR == xDriverEnumBoards(hDriver, ulBoard, sizeof(tBoardInfo), &tBoardInfo))
  {
    uint32_t            ulChannel    = 0;
    CHANNEL_INFORMATION tChannelInfo = {0};

    printf("Found Board %.10s\r\n", tBoardInfo.abBoardName);
    if(strlen( (char*)tBoardInfo.abBoardAlias) != 0)
      printf(" Alias        : %.10s\r\n", tBoardInfo.abBoardAlias);

    printf(" DeviceNumber : %u\r\n", tBoardInfo.tSystemInfo.ulDeviceNumber);
    printf(" SerialNumber : %u\r\n", tBoardInfo.tSystemInfo.ulSerialNumber);
    printf(" Board ID     : %u\r\n", tBoardInfo.ulBoardID);
    printf(" System Error : 0x%08X\r\n", tBoardInfo.ulSystemError);
    printf(" Channels     : %u\r\n", tBoardInfo.ulChannelCnt);
    printf(" DPM Size     : %u\r\n", tBoardInfo.ulDpmTotalSize);

    /* iterate over all channels on the current board */
    while(CIFX_NO_ERROR == xDriverEnumChannels(hDriver, ulBoard, ulChannel, sizeof(tChannelInfo), &tChannelInfo))
    {
      printf(" - Channel %u:\r\n", ulChannel);
      printf("    Firmware : %s\r\n", tChannelInfo.abFWName);
      printf("    Version  : %u.%u.%u build %u\r\n", 
             tChannelInfo.usFWMajor,
             tChannelInfo.usFWMinor,
             tChannelInfo.usFWBuild,
             tChannelInfo.usFWRevision);
      printf("    Date     : %02u/%02u/%04u\r\n", 
             tChannelInfo.bFWMonth,
             tChannelInfo.bFWDay,
             tChannelInfo.usFWYear);

      ++ulChannel;
    }

    ++ulBoard;
  }
  printf("----------------------------------------------------\r\n");
}

/*****************************************************************************/
/*! Function to demonstrate system device functionality (Packet Transfer)
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
long SysdeviceDemo(HANDLE hDriver, char* szBoard)
{
  long   lRet = CIFX_NO_ERROR;
  HANDLE hSys = NULL;

  printf("---------- System Device handling demo ----------\r\n");

  /* Driver/Toolkit successfully opened */
  lRet = xSysdeviceOpen(hDriver, szBoard, &hSys);

  if(CIFX_NO_ERROR != lRet)
  {
    printf("Error opening SystemDevice!\r\n");

  } else
  {
    SYSTEM_CHANNEL_SYSTEM_INFO_BLOCK tSysInfo       = {0};
    uint32_t                         ulSendPktCount = 0;
    uint32_t                         ulRecvPktCount = 0;
    CIFX_PACKET                      tSendPkt       = {0};
    CIFX_PACKET                      tRecvPkt       = {0};
   
    /* System channel successfully opened, try to read the System Info Block */
    if( CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSys, CIFX_INFO_CMD_SYSTEM_INFO_BLOCK, sizeof(tSysInfo), &tSysInfo)))
    {
      printf("Error querying system information block\r\n");
    } else
    {
      printf("System Channel Info Block:\r\n");
      printf("DPM Size         : %u\r\n", tSysInfo.ulDpmTotalSize);
      printf("Device Number    : %u\r\n", tSysInfo.ulDeviceNumber);
      printf("Serial Number    : %u\r\n", tSysInfo.ulSerialNumber);
      printf("Manufacturer     : %u\r\n", tSysInfo.usManufacturer);
      printf("Production Date  : %u\r\n", tSysInfo.usProductionDate);
      printf("Device Class     : %u\r\n", tSysInfo.usDeviceClass);
      printf("HW Revision      : %u\r\n", tSysInfo.bHwRevision);
      printf("HW Compatibility : %u\r\n", tSysInfo.bHwCompatibility);
    }

    /* Do a simple Packet exchange via system channel */
    xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
    printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
           ulSendPktCount, ulRecvPktCount);

    if(CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSys, &tSendPkt, PACKET_WAIT_TIMEOUT)))
    {
      printf("Error sending packet to device!\r\n");
    } else
    {
      printf("Send Packet:\r\n");
      DumpPacket(&tSendPkt);

      xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
      printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
            ulSendPktCount, ulRecvPktCount);

      if(CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSys, sizeof(tRecvPkt), &tRecvPkt, PACKET_WAIT_TIMEOUT)) )
      {
        printf("Error getting packet from device!\r\n");
      } else
      {
        printf("Received Packet:\r\n");
        DumpPacket(&tRecvPkt);

        xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
        printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
              ulSendPktCount, ulRecvPktCount);
      }
    }
    xSysdeviceClose(hSys);
  }

  printf(" State = 0x%08X\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
long ChannelDemo(HANDLE hDriver, char* szBoard, uint32_t ulChannel)
{
  HANDLE hChannel = NULL;
  long   lRet     = CIFX_NO_ERROR;

  printf("---------- Communication Channel demo ----------\r\n");

  lRet = xChannelOpen(hDriver, szBoard, ulChannel, &hChannel);

  if(CIFX_NO_ERROR != lRet)
  {
    printf("Error opening Channel!");

  } else
  {
    CHANNEL_INFORMATION tChannelInfo  = {0};
    CIFX_PACKET         tSendPkt      = {0};
    CIFX_PACKET         tRecvPkt      = {0};
    /* Read and write I/O data (32Bytes). Output data will be incremented each cyle */
    unsigned char       abSendData[32] = {0};
    unsigned char       abRecvData[32] = {0};
    unsigned long       ulCycle        = 0;
    uint32_t            ulState        = 0;

    /* Channel successfully opened, so query basic information */
    if( CIFX_NO_ERROR != (lRet = xChannelInfo(hChannel, sizeof(CHANNEL_INFORMATION), &tChannelInfo)))
    {
      printf("Error querying system information block\r\n");
    } else
    {
      printf("Communication Channel Info:\r\n");
      printf("Device Number    : %u\r\n", tChannelInfo.ulDeviceNumber);
      printf("Serial Number    : %u\r\n", tChannelInfo.ulSerialNumber);
      printf("Firmware         : %s\r\n", tChannelInfo.abFWName);
      printf("FW Version       : %u.%u.%u build %u\r\n", 
              tChannelInfo.usFWMajor,
              tChannelInfo.usFWMinor,
              tChannelInfo.usFWBuild,
              tChannelInfo.usFWRevision);
      printf("FW Date          : %02u/%02u/%04u\r\n", 
              tChannelInfo.bFWMonth,
              tChannelInfo.bFWDay,
              tChannelInfo.usFWYear);

      printf("Mailbox Size     : %u\r\n", tChannelInfo.ulMailboxSize);
    }

    /* Do a basic Packet Transfer */
    if(CIFX_NO_ERROR != (lRet = xChannelPutPacket(hChannel, &tSendPkt, PACKET_WAIT_TIMEOUT)))
    {
      printf("Error sending packet to device!\r\n");
    } else
    {
      printf("Send Packet:\r\n");
      DumpPacket(&tSendPkt);
                                       
      if(CIFX_NO_ERROR != (lRet = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, PACKET_WAIT_TIMEOUT)) )
      {
        printf("Error getting packet from device!\r\n");
      } else
      {
        printf("Received Packet:\r\n");
        DumpPacket(&tRecvPkt);
      }
    }

    /* Set Host Ready to signal the filed bus an application is ready */
    lRet = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, HOSTSTATE_TIMEOUT);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error setting host ready!\r\n");
    } else
    {
      /* Switch on the bus if it is not automatically running (see configuration options) */
      lRet = xChannelBusState( hChannel, CIFX_BUS_STATE_ON, &ulState, 0L);

      if(CIFX_NO_ERROR != lRet)
      {
        printf("Unable to start the filed bus!\r\n");
      } else
      {
        /* Do I/O Data exchange until a key is hit */
        while(!kbhit())
        {
          if(CIFX_NO_ERROR != (lRet = xChannelIORead(hChannel, 0, 0, sizeof(abRecvData), abRecvData, IO_WAIT_TIMEOUT)))
          {
            printf("Error reading IO Data area!\r\n");
            break;
          } else
          {
            printf("IORead Data:");
            DumpData(abRecvData, sizeof(abRecvData));

            if(CIFX_NO_ERROR != (lRet = xChannelIOWrite(hChannel, 0, 0, sizeof(abRecvData), abRecvData, IO_WAIT_TIMEOUT)))
            {
              printf("Error writing to IO Data area!\r\n");
              break;
            } else
            {
              printf("IOWrite Data:");
              DumpData(abSendData, sizeof(abSendData));

              memset(abSendData, ulCycle + 1, sizeof(abSendData));
            }
          }
        }
      }
    }

    /* Switch off the bus  */
    lRet = xChannelBusState( hChannel, CIFX_BUS_STATE_OFF, &ulState, 0L);

    /* Set Host not ready to stop bus communication */
    xChannelHostState(hChannel, CIFX_HOST_STATE_NOT_READY, &ulState, HOSTSTATE_TIMEOUT);

    /* Close the communication channel */
    xChannelClose(hChannel);
  }

  if(CIFX_NO_ERROR != lRet)
  {
    char szBuffer[256] = {0};
    xDriverGetErrorDescription(lRet, szBuffer, sizeof(szBuffer));

    printf(" State = 0x%08X <%s>\r\n", lRet, szBuffer);
  } else
  {
    printf(" State = 0x%08X\r\n", lRet);
  }
  printf("----------------------------------------------------\r\n");

  return lRet;
}


/*****************************************************************************/
/*! The main function 
*   \return 0 on success                                                     */
/*****************************************************************************/
int main(int argc, char* argv[])
{
  HANDLE hDriver  = NULL;
  long lRet = CIFX_NO_ERROR; 

  UNREFERENCED_PARAMETER(argc);
  UNREFERENCED_PARAMETER(argv);

  /* Open the cifX driver */
  lRet = xDriverOpen(&hDriver);

  if(CIFX_NO_ERROR != lRet)
  {
    printf("Error opening driver. lRet=0x%08X\r\n", lRet);
  } else
  {
    /* Example how to find a cifX/comX board */
    EnumBoardDemo(hDriver);

    /* Example how to communicate with the SYSTEM device of a board */
    SysdeviceDemo(hDriver, "cifX0");

    /* Example how to communicate with a communication channel on a board */
    ChannelDemo(hDriver, "cifX0", 0);

    /* Close the cifX driver */
    xDriverClose(hDriver);
  }

	return 0;
}
